package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link Config}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableConfig.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableConfig implements Config {
  private final String id;
  private final Version version;
  private final Date createdAt;
  private final Date updatedAt;
  private final ConfigSpec configSpec;

  private ImmutableConfig(
      String id,
      Version version,
      Date createdAt,
      Date updatedAt,
      ConfigSpec configSpec) {
    this.id = id;
    this.version = version;
    this.createdAt = createdAt;
    this.updatedAt = updatedAt;
    this.configSpec = configSpec;
  }

  /**
   * @return The value of the {@code id} attribute
   */
  @JsonProperty("ID")
  @Override
  public String id() {
    return id;
  }

  /**
   * @return The value of the {@code version} attribute
   */
  @JsonProperty("Version")
  @Override
  public Version version() {
    return version;
  }

  /**
   * @return The value of the {@code createdAt} attribute
   */
  @JsonProperty("CreatedAt")
  @Override
  public Date createdAt() {
    return createdAt;
  }

  /**
   * @return The value of the {@code updatedAt} attribute
   */
  @JsonProperty("UpdatedAt")
  @Override
  public Date updatedAt() {
    return updatedAt;
  }

  /**
   * @return The value of the {@code configSpec} attribute
   */
  @JsonProperty("Spec")
  @Override
  public ConfigSpec configSpec() {
    return configSpec;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Config#id() id} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for id
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableConfig withId(String value) {
    String newValue = Objects.requireNonNull(value, "id");
    if (this.id.equals(newValue)) return this;
    return new ImmutableConfig(newValue, this.version, this.createdAt, this.updatedAt, this.configSpec);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Config#version() version} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for version
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableConfig withVersion(Version value) {
    if (this.version == value) return this;
    Version newValue = Objects.requireNonNull(value, "version");
    return new ImmutableConfig(this.id, newValue, this.createdAt, this.updatedAt, this.configSpec);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Config#createdAt() createdAt} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for createdAt
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableConfig withCreatedAt(Date value) {
    if (this.createdAt == value) return this;
    Date newValue = Objects.requireNonNull(value, "createdAt");
    return new ImmutableConfig(this.id, this.version, newValue, this.updatedAt, this.configSpec);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Config#updatedAt() updatedAt} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for updatedAt
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableConfig withUpdatedAt(Date value) {
    if (this.updatedAt == value) return this;
    Date newValue = Objects.requireNonNull(value, "updatedAt");
    return new ImmutableConfig(this.id, this.version, this.createdAt, newValue, this.configSpec);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Config#configSpec() configSpec} attribute.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for configSpec
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableConfig withConfigSpec(ConfigSpec value) {
    if (this.configSpec == value) return this;
    ConfigSpec newValue = Objects.requireNonNull(value, "configSpec");
    return new ImmutableConfig(this.id, this.version, this.createdAt, this.updatedAt, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableConfig} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableConfig
        && equalsByValue((ImmutableConfig) another);
  }

  private boolean equalsByValue(ImmutableConfig another) {
    return id.equals(another.id)
        && version.equals(another.version)
        && createdAt.equals(another.createdAt)
        && updatedAt.equals(another.updatedAt)
        && configSpec.equals(another.configSpec);
  }

  /**
   * Computes a hash code from attributes: {@code id}, {@code version}, {@code createdAt}, {@code updatedAt}, {@code configSpec}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + id.hashCode();
    h += (h << 5) + version.hashCode();
    h += (h << 5) + createdAt.hashCode();
    h += (h << 5) + updatedAt.hashCode();
    h += (h << 5) + configSpec.hashCode();
    return h;
  }

  /**
   * Prints the immutable value {@code Config} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "Config{"
        + "id=" + id
        + ", version=" + version
        + ", createdAt=" + createdAt
        + ", updatedAt=" + updatedAt
        + ", configSpec=" + configSpec
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link Config} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable Config instance
   */
  public static ImmutableConfig copyOf(Config instance) {
    if (instance instanceof ImmutableConfig) {
      return (ImmutableConfig) instance;
    }
    return ImmutableConfig.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableConfig ImmutableConfig}.
   * <pre>
   * ImmutableConfig.builder()
   *    .id(String) // required {@link Config#id() id}
   *    .version(org.mandas.docker.client.messages.swarm.Version) // required {@link Config#version() version}
   *    .createdAt(Date) // required {@link Config#createdAt() createdAt}
   *    .updatedAt(Date) // required {@link Config#updatedAt() updatedAt}
   *    .configSpec(org.mandas.docker.client.messages.swarm.ConfigSpec) // required {@link Config#configSpec() configSpec}
   *    .build();
   * </pre>
   * @return A new ImmutableConfig builder
   */
  public static ImmutableConfig.Builder builder() {
    return new ImmutableConfig.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableConfig ImmutableConfig}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private static final long INIT_BIT_ID = 0x1L;
    private static final long INIT_BIT_VERSION = 0x2L;
    private static final long INIT_BIT_CREATED_AT = 0x4L;
    private static final long INIT_BIT_UPDATED_AT = 0x8L;
    private static final long INIT_BIT_CONFIG_SPEC = 0x10L;
    private long initBits = 0x1fL;

    private String id;
    private Version version;
    private Date createdAt;
    private Date updatedAt;
    private ConfigSpec configSpec;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code Config} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(Config instance) {
      Objects.requireNonNull(instance, "instance");
      this.id(instance.id());
      this.version(instance.version());
      this.createdAt(instance.createdAt());
      this.updatedAt(instance.updatedAt());
      this.configSpec(instance.configSpec());
      return this;
    }

    /**
     * Initializes the value for the {@link Config#id() id} attribute.
     * @param id The value for id 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("ID")
    public final Builder id(String id) {
      this.id = Objects.requireNonNull(id, "id");
      initBits &= ~INIT_BIT_ID;
      return this;
    }

    /**
     * Initializes the value for the {@link Config#version() version} attribute.
     * @param version The value for version 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Version")
    public final Builder version(Version version) {
      this.version = Objects.requireNonNull(version, "version");
      initBits &= ~INIT_BIT_VERSION;
      return this;
    }

    /**
     * Initializes the value for the {@link Config#createdAt() createdAt} attribute.
     * @param createdAt The value for createdAt 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("CreatedAt")
    public final Builder createdAt(Date createdAt) {
      this.createdAt = Objects.requireNonNull(createdAt, "createdAt");
      initBits &= ~INIT_BIT_CREATED_AT;
      return this;
    }

    /**
     * Initializes the value for the {@link Config#updatedAt() updatedAt} attribute.
     * @param updatedAt The value for updatedAt 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("UpdatedAt")
    public final Builder updatedAt(Date updatedAt) {
      this.updatedAt = Objects.requireNonNull(updatedAt, "updatedAt");
      initBits &= ~INIT_BIT_UPDATED_AT;
      return this;
    }

    /**
     * Initializes the value for the {@link Config#configSpec() configSpec} attribute.
     * @param configSpec The value for configSpec 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Spec")
    public final Builder configSpec(ConfigSpec configSpec) {
      this.configSpec = Objects.requireNonNull(configSpec, "configSpec");
      initBits &= ~INIT_BIT_CONFIG_SPEC;
      return this;
    }

    /**
     * Builds a new {@link ImmutableConfig ImmutableConfig}.
     * @return An immutable instance of Config
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableConfig build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableConfig(id, version, createdAt, updatedAt, configSpec);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_ID) != 0) attributes.add("id");
      if ((initBits & INIT_BIT_VERSION) != 0) attributes.add("version");
      if ((initBits & INIT_BIT_CREATED_AT) != 0) attributes.add("createdAt");
      if ((initBits & INIT_BIT_UPDATED_AT) != 0) attributes.add("updatedAt");
      if ((initBits & INIT_BIT_CONFIG_SPEC) != 0) attributes.add("configSpec");
      return "Cannot build Config, some of required attributes are not set " + attributes;
    }
  }

  /**
   * Immutable implementation of {@link Config.Criteria}.
   * <p>
   * Use the builder to create immutable instances:
   * {@code ImmutableConfig.Criteria.builder()}.
   */
  static final class Criteria implements Config.Criteria {
    private final @Nullable String configId;
    private final @Nullable String label;
    private final @Nullable String name;

    private Criteria(
        @Nullable String configId,
        @Nullable String label,
        @Nullable String name) {
      this.configId = configId;
      this.label = label;
      this.name = name;
    }

    /**
     * @return The value of the {@code configId} attribute
     */
    @Override
    public @Nullable String configId() {
      return configId;
    }

    /**
     * @return The value of the {@code label} attribute
     */
    @Override
    public @Nullable String label() {
      return label;
    }

    /**
     * @return The value of the {@code name} attribute
     */
    @Override
    public @Nullable String name() {
      return name;
    }

    /**
     * Copy the current immutable object by setting a value for the {@link Config.Criteria#configId() configId} attribute.
     * An equals check used to prevent copying of the same value by returning {@code this}.
     * @param value A new value for configId (can be {@code null})
     * @return A modified copy or the {@code this} object
     */
    public final ImmutableConfig.Criteria withConfigId(@Nullable String value) {
      if (Objects.equals(this.configId, value)) return this;
      return new ImmutableConfig.Criteria(value, this.label, this.name);
    }

    /**
     * Copy the current immutable object by setting a value for the {@link Config.Criteria#label() label} attribute.
     * An equals check used to prevent copying of the same value by returning {@code this}.
     * @param value A new value for label (can be {@code null})
     * @return A modified copy or the {@code this} object
     */
    public final ImmutableConfig.Criteria withLabel(@Nullable String value) {
      if (Objects.equals(this.label, value)) return this;
      return new ImmutableConfig.Criteria(this.configId, value, this.name);
    }

    /**
     * Copy the current immutable object by setting a value for the {@link Config.Criteria#name() name} attribute.
     * An equals check used to prevent copying of the same value by returning {@code this}.
     * @param value A new value for name (can be {@code null})
     * @return A modified copy or the {@code this} object
     */
    public final ImmutableConfig.Criteria withName(@Nullable String value) {
      if (Objects.equals(this.name, value)) return this;
      return new ImmutableConfig.Criteria(this.configId, this.label, value);
    }

    /**
     * This instance is equal to all instances of {@code Criteria} that have equal attribute values.
     * @return {@code true} if {@code this} is equal to {@code another} instance
     */
    @Override
    public boolean equals(Object another) {
      if (this == another) return true;
      return another instanceof ImmutableConfig.Criteria
          && equalsByValue((ImmutableConfig.Criteria) another);
    }

    private boolean equalsByValue(ImmutableConfig.Criteria another) {
      return Objects.equals(configId, another.configId)
          && Objects.equals(label, another.label)
          && Objects.equals(name, another.name);
    }

    /**
     * Computes a hash code from attributes: {@code configId}, {@code label}, {@code name}.
     * @return hashCode value
     */
    @Override
    public int hashCode() {
      int h = 5381;
      h += (h << 5) + Objects.hashCode(configId);
      h += (h << 5) + Objects.hashCode(label);
      h += (h << 5) + Objects.hashCode(name);
      return h;
    }

    /**
     * Prints the immutable value {@code Criteria} with attribute values.
     * @return A string representation of the value
     */
    @Override
    public String toString() {
      return "Criteria{"
          + "configId=" + configId
          + ", label=" + label
          + ", name=" + name
          + "}";
    }

    /**
     * Creates an immutable copy of a {@link Config.Criteria} value.
     * Uses accessors to get values to initialize the new immutable instance.
     * If an instance is already immutable, it is returned as is.
     * @param instance The instance to copy
     * @return A copied immutable Criteria instance
     */
    public static ImmutableConfig.Criteria copyOf(Config.Criteria instance) {
      if (instance instanceof ImmutableConfig.Criteria) {
        return (ImmutableConfig.Criteria) instance;
      }
      return ImmutableConfig.Criteria.builder()
          .from(instance)
          .build();
    }

    /**
     * Creates a builder for {@link ImmutableConfig.Criteria Criteria}.
     * <pre>
     * ImmutableConfig.Criteria.builder()
     *    .configId(String | null) // nullable {@link Config.Criteria#configId() configId}
     *    .label(String | null) // nullable {@link Config.Criteria#label() label}
     *    .name(String | null) // nullable {@link Config.Criteria#name() name}
     *    .build();
     * </pre>
     * @return A new Criteria builder
     */
    public static ImmutableConfig.Criteria.Builder builder() {
      return new ImmutableConfig.Criteria.Builder();
    }

    /**
     * Builds instances of type {@link ImmutableConfig.Criteria Criteria}.
     * Initialize attributes and then invoke the {@link #build()} method to create an
     * immutable instance.
     * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
     * but instead used immediately to create instances.</em>
     */
    static final class Builder implements Config.Criteria.Builder {
      private String configId;
      private String label;
      private String name;

      private Builder() {
      }

      /**
       * Fill a builder with attribute values from the provided {@code Criteria} instance.
       * Regular attribute values will be replaced with those from the given instance.
       * Absent optional values will not replace present values.
       * @param instance The instance from which to copy values
       * @return {@code this} builder for use in a chained invocation
       */
      public final Builder from(Config.Criteria instance) {
        Objects.requireNonNull(instance, "instance");
        @Nullable String configIdValue = instance.configId();
        if (configIdValue != null) {
          configId(configIdValue);
        }
        @Nullable String labelValue = instance.label();
        if (labelValue != null) {
          label(labelValue);
        }
        @Nullable String nameValue = instance.name();
        if (nameValue != null) {
          name(nameValue);
        }
        return this;
      }

      /**
       * Initializes the value for the {@link Config.Criteria#configId() configId} attribute.
       * @param configId The value for configId (can be {@code null})
       * @return {@code this} builder for use in a chained invocation
       */
      public final Builder configId(@Nullable String configId) {
        this.configId = configId;
        return this;
      }

      /**
       * Initializes the value for the {@link Config.Criteria#label() label} attribute.
       * @param label The value for label (can be {@code null})
       * @return {@code this} builder for use in a chained invocation
       */
      public final Builder label(@Nullable String label) {
        this.label = label;
        return this;
      }

      /**
       * Initializes the value for the {@link Config.Criteria#name() name} attribute.
       * @param name The value for name (can be {@code null})
       * @return {@code this} builder for use in a chained invocation
       */
      public final Builder name(@Nullable String name) {
        this.name = name;
        return this;
      }

      /**
       * Builds a new {@link ImmutableConfig.Criteria Criteria}.
       * @return An immutable instance of Criteria
       * @throws java.lang.IllegalStateException if any required attributes are missing
       */
      public ImmutableConfig.Criteria build() {
        return new ImmutableConfig.Criteria(configId, label, name);
      }
    }
  }
}
