/*******************************************************************************
 * Copyright (c) 2005, 2017 IBM Corporation and others.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 * 
 * SPDX-License-Identifier: EPL-2.0
 *
 *******************************************************************************/
package org.eclipse.dltk.dbgp.internal;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;

import org.eclipse.dltk.dbgp.IDbgpRawPacket;
import org.eclipse.dltk.dbgp.exceptions.DbgpException;
import org.eclipse.dltk.dbgp.internal.utils.DbgpXmlParser;
import org.eclipse.dltk.debug.core.DLTKDebugPlugin;
import org.eclipse.osgi.util.NLS;
import org.w3c.dom.Document;

public class DbgpRawPacket implements IDbgpRawPacket {

	protected static int readPacketSize(InputStream input) throws IOException {
		int size = 0;
		for (;;) {
			int b = input.read();
			if (b == -1) {
				throw new IOException();
			}
			if (b == 0) {
				break;
			}
			if (b >= '0' && b <= '9') {
				size = size * 10 + (b - '0');
			} else {
				final String msg = NLS.bind(
						Messages.DbgpRawPacket_invalidCharInPacketSize,
						Integer.toString(b));
				DLTKDebugPlugin.logWarning(msg);
				throw new IOException(msg);
			}
		}
		if (size == 0) {
			throw new IOException(Messages.DbgpRawPacket_zeroPacketSize);
		}
		return size;
	}

	protected static byte[] readPacketXml(InputStream input, int size)
			throws IOException {
		byte[] bytes = new byte[size];

		int offset = 0;
		int n;
		while ((offset < size)
				&& (n = input.read(bytes, offset, size - offset)) != -1) {
			offset += n;
		}

		if (offset != size) {
			throw new IOException(Messages.DbgpRawPacket_cantReadPacketBody);
		}

		if (input.read() != 0) {
			throw new IOException(Messages.DbgpRawPacket_noTerminationByte);
		}

		return bytes;
	}

	public static DbgpRawPacket readPacket(InputStream input)
			throws IOException {
		int size = readPacketSize(input);
		byte[] xml = readPacketXml(input, size);
		return new DbgpRawPacket(size, xml);
	}

	private final int size;

	private final byte[] xml;

	public DbgpRawPacket(int size, byte[] xml) {
		this.size = size;
		this.xml = xml;
	}

	public int getSize() {
		return size;
	}

	public byte[] getXml() {
		return xml;
	}

	public Document getParsedXml() throws DbgpException {
		return DbgpXmlParser.parseXml(xml);
	}

	@Override
	public String toString() {
		return "DbgpPacket (" + size + " bytes) " + xml; //$NON-NLS-1$ //$NON-NLS-2$
	}

	@Override
	public String getPacketAsString() {
		return new String(xml, StandardCharsets.US_ASCII);
	}
}
