/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.rdf4j.sail.nativerdf.btree;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.StandardOpenOption;
import java.util.Arrays;
import java.util.BitSet;
import org.eclipse.rdf4j.common.io.ByteArrayUtil;
import org.eclipse.rdf4j.sail.nativerdf.btree.BTree;
import org.eclipse.rdf4j.sail.nativerdf.btree.Node;

class AllocatedNodesList
implements Closeable {
    private static final byte[] MAGIC_NUMBER = new byte[]{97, 110, 102};
    private static final byte FILE_FORMAT_VERSION = 1;
    private static final int HEADER_LENGTH = MAGIC_NUMBER.length + 1;
    private final BTree btree;
    private final File allocNodesFile;
    private final FileChannel channel;
    private MappedByteBuffer mapped;
    private int bitCapacity = 0;
    private BitSet allocatedNodes;
    private boolean needsSync = false;
    private final boolean forceSync;

    public AllocatedNodesList(File allocNodesFile, BTree btree, boolean forceSync) throws IOException {
        if (allocNodesFile == null) {
            throw new IllegalArgumentException("allocNodesFile must not be null");
        }
        if (btree == null) {
            throw new IllegalArgumentException("btree muts not be null");
        }
        this.allocNodesFile = allocNodesFile;
        this.btree = btree;
        this.forceSync = forceSync;
        this.channel = FileChannel.open(allocNodesFile.toPath(), StandardOpenOption.READ, StandardOpenOption.WRITE, StandardOpenOption.CREATE);
        this.mapped = null;
        this.bitCapacity = 64;
    }

    public File getFile() {
        return this.allocNodesFile;
    }

    @Override
    public synchronized void close() throws IOException {
        this.close(true);
    }

    public synchronized boolean delete() throws IOException {
        this.close(false);
        return this.allocNodesFile.delete();
    }

    public synchronized void close(boolean syncChanges) throws IOException {
        if (syncChanges) {
            this.sync();
        }
        this.allocatedNodes = null;
        this.needsSync = false;
        this.mapped = null;
        this.channel.close();
    }

    public synchronized void sync() throws IOException {
        if (!this.needsSync) {
            return;
        }
        if (this.mapped != null && this.forceSync) {
            this.mapped.force();
        }
        this.needsSync = false;
    }

    private void scheduleSync() {
        if (!this.needsSync) {
            this.needsSync = true;
        }
    }

    public synchronized void clear() throws IOException {
        this.initAllocatedNodes();
        this.allocatedNodes.clear();
        if (this.mapped != null && this.bitCapacity > 0) {
            int byteCount = this.bitCapacity + 7 >>> 3;
            int start = HEADER_LENGTH;
            int end = start + byteCount;
            for (int pos = start; pos < end; ++pos) {
                this.mapped.put(pos, (byte)0);
            }
        }
        this.scheduleSync();
    }

    public synchronized int allocateNode() throws IOException {
        this.initAllocatedNodes();
        int newNodeID = this.allocatedNodes.nextClearBit(1);
        this.allocatedNodes.set(newNodeID);
        this.ensureCapacityForBit(newNodeID);
        this.setOnDiskBit(newNodeID, true);
        this.scheduleSync();
        return newNodeID;
    }

    public synchronized void freeNode(int nodeID) throws IOException {
        this.initAllocatedNodes();
        this.allocatedNodes.clear(nodeID);
        if (this.bitCapacity > 0 && nodeID < this.bitCapacity && this.mapped != null) {
            this.setOnDiskBit(nodeID, false);
        }
        this.scheduleSync();
    }

    public synchronized int getMaxNodeID() throws IOException {
        this.initAllocatedNodes();
        return Math.max(0, this.allocatedNodes.length() - 1);
    }

    public synchronized int getNodeCount() throws IOException {
        this.initAllocatedNodes();
        return this.allocatedNodes.cardinality();
    }

    private void initAllocatedNodes() throws IOException {
        if (this.allocatedNodes != null) {
            return;
        }
        long size = this.channel.size();
        if (size > 0L) {
            this.loadAllocatedNodesInfo();
        } else {
            this.crawlAllocatedNodes();
        }
        this.remapFromAllocatedNodes();
    }

    private void loadAllocatedNodesInfo() throws IOException {
        byte[] data;
        long size = this.channel.size();
        if (size <= 0L) {
            this.allocatedNodes = new BitSet();
            return;
        }
        ByteBuffer buf = ByteBuffer.allocate((int)size);
        this.channel.position(0L);
        while (buf.hasRemaining() && this.channel.read(buf) >= 0) {
        }
        byte[] fileBytes = buf.array();
        if (size >= (long)HEADER_LENGTH && this.hasMagicHeader(fileBytes)) {
            byte version = fileBytes[MAGIC_NUMBER.length];
            if (version > 1) {
                throw new IOException("Unable to read allocated nodes file; it uses a newer file format");
            }
            if (version != 1) {
                throw new IOException("Unable to read allocated nodes file; invalid file format version: " + version);
            }
            int dataLength = (int)(size - (long)HEADER_LENGTH);
            data = new byte[dataLength];
            System.arraycopy(fileBytes, HEADER_LENGTH, data, 0, dataLength);
        } else {
            data = fileBytes;
            this.scheduleSync();
        }
        this.allocatedNodes = ByteArrayUtil.toBitSet((byte[])data);
    }

    private boolean hasMagicHeader(byte[] fileBytes) {
        if (fileBytes.length < MAGIC_NUMBER.length) {
            return false;
        }
        for (int i = 0; i < MAGIC_NUMBER.length; ++i) {
            if (fileBytes[i] == MAGIC_NUMBER[i]) continue;
            return false;
        }
        return true;
    }

    private void crawlAllocatedNodes() throws IOException {
        this.allocatedNodes = new BitSet();
        Node rootNode = this.btree.readRootNode();
        if (rootNode != null) {
            this.crawlAllocatedNodes(rootNode);
        }
        this.scheduleSync();
    }

    private void crawlAllocatedNodes(Node node) throws IOException {
        try {
            this.allocatedNodes.set(node.getID());
            if (!node.isLeaf()) {
                for (int i = 0; i < node.getValueCount() + 1; ++i) {
                    this.crawlAllocatedNodes(node.getChildNode(i));
                }
            }
        }
        finally {
            node.release();
        }
    }

    private void ensureCapacityForBit(int bitIndex) throws IOException {
        int neededBytes;
        byte[] data;
        int neededBits = bitIndex + 1;
        if (neededBits <= this.bitCapacity && this.mapped != null) {
            return;
        }
        int newBitCapacity = Math.max(neededBits, this.bitCapacity);
        newBitCapacity = newBitCapacity + 32768 - 1 & Short.MIN_VALUE;
        assert ((newBitCapacity -= HEADER_LENGTH * 8) > 0);
        if (newBitCapacity < 0) {
            newBitCapacity = neededBits + 8;
        }
        if ((data = ByteArrayUtil.toByteArray((BitSet)this.allocatedNodes)).length < (neededBytes = newBitCapacity + 7 >>> 3)) {
            data = Arrays.copyOf(data, neededBytes);
        }
        long newFileSize = (long)HEADER_LENGTH + (long)data.length;
        long currentSize = this.channel.size();
        if (currentSize < newFileSize) {
            this.channel.position(newFileSize - 1L);
            this.channel.write(ByteBuffer.wrap(new byte[]{0}));
        } else if (currentSize > newFileSize) {
            this.channel.truncate(newFileSize);
        }
        this.mapped = this.channel.map(FileChannel.MapMode.READ_WRITE, 0L, newFileSize);
        this.mapped.position(0);
        this.mapped.put(MAGIC_NUMBER);
        this.mapped.put((byte)1);
        this.mapped.put(data);
        this.bitCapacity = newBitCapacity;
    }

    private void remapFromAllocatedNodes() throws IOException {
        int neededBytes;
        byte[] data;
        int neededBits = Math.max(this.allocatedNodes.length(), 1);
        int newBitCapacity = neededBits + 32768 - 1 & Short.MIN_VALUE;
        assert ((newBitCapacity -= HEADER_LENGTH * 8) > 0);
        if (newBitCapacity < 0) {
            newBitCapacity = neededBits + 8;
        }
        if ((data = ByteArrayUtil.toByteArray((BitSet)this.allocatedNodes)).length < (neededBytes = newBitCapacity + 7 >>> 3)) {
            data = Arrays.copyOf(data, neededBytes);
        }
        long newFileSize = (long)HEADER_LENGTH + (long)data.length;
        this.channel.truncate(newFileSize);
        this.channel.position(newFileSize - 1L);
        this.channel.write(ByteBuffer.wrap(new byte[]{0}));
        this.mapped = this.channel.map(FileChannel.MapMode.READ_WRITE, 0L, newFileSize);
        this.mapped.position(0);
        this.mapped.put(MAGIC_NUMBER);
        this.mapped.put((byte)1);
        this.mapped.put(data);
        this.bitCapacity = newBitCapacity;
    }

    private void setOnDiskBit(int bitIndex, boolean value) {
        if (this.mapped == null || bitIndex < 0) {
            return;
        }
        int byteIndex = bitIndex >>> 3;
        int bitInByte = bitIndex & 7;
        int fileOffset = HEADER_LENGTH + byteIndex;
        if (fileOffset >= this.mapped.capacity()) {
            return;
        }
        byte b = this.mapped.get(fileOffset);
        int mask = 1 << bitInByte;
        b = value ? (byte)(b | mask) : (byte)(b & ~mask);
        this.mapped.put(fileOffset, b);
    }
}

