/*
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.lsat.timing;

import java.util.Map;

import javax.script.ScriptException;

import org.mozilla.javascript.Context;
import org.mozilla.javascript.Scriptable;
import org.mozilla.javascript.ScriptableObject;

/**
 * An Engine that executes JavaScript using Rhino.
 */
public class JavaScriptEngine {

    public Object eval(String script, Map<String, Object> args) throws ScriptException {
        Context context = Context.enter();
        try {
            // Initialize standard objects and scope
            Scriptable scope = context.initStandardObjects();

            // Add all arguments to the scope
            for (Map.Entry<String, Object> entry : args.entrySet()) {
                Object wrappedValue = Context.javaToJS(entry.getValue(), scope);
                ScriptableObject.putProperty(scope, entry.getKey(), wrappedValue);
            }

            // Evaluate the script
            Object result = context.evaluateString(scope, script, "script", 1, null);

            // Extract all variables from scope back to args map
            for (Object key : ScriptableObject.getPropertyIds(scope)) {
                if (key instanceof String propertyName) {
                    // Skip internal properties that start with underscore
                    if (!propertyName.startsWith("_")) {
                        Object value = ScriptableObject.getProperty(scope, propertyName);
                        // Convert JavaScript value to Java
                        Object javaValue = Context.jsToJava(value, Object.class);
                        args.put(propertyName, javaValue);
                    }
                }
            }

            // Convert result to Java object
            return Context.jsToJava(result, Object.class);

        } catch (Exception e) {
            throw new ScriptException(e);
        } finally {
            Context.exit();
        }
    }
}
