/**
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 * 
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.lsat.scheduler.etfgen;

import activity.Activity;
import activity.ResourceAction;
import activity.TracePoint;
import com.google.common.collect.Iterables;
import dispatching.Attribute;
import dispatching.Dispatch;
import dispatching.HasUserAttributes;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Function;
import lsat_graph.ActionTask;
import lsat_graph.Bounds;
import lsat_graph.ClaimReleaseResource;
import lsat_graph.ClaimTask;
import lsat_graph.ClaimedByScheduledTask;
import lsat_graph.DispatchGraph;
import lsat_graph.DispatchGroupResource;
import lsat_graph.DispatchGroupTask;
import lsat_graph.EventAnnotation;
import lsat_graph.PeripheralActionTask;
import lsat_graph.PeripheralResource;
import lsat_graph.ReleaseTask;
import lsat_graph.StochasticAnnotation;
import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.EMap;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.lsat.common.graph.directed.Aspect;
import org.eclipse.lsat.common.graph.directed.DirectedGraph;
import org.eclipse.lsat.common.graph.directed.Edge;
import org.eclipse.lsat.common.graph.directed.Node;
import org.eclipse.lsat.common.scheduler.graph.Constraint;
import org.eclipse.lsat.common.scheduler.graph.Dependency;
import org.eclipse.lsat.common.scheduler.graph.JitConstraint;
import org.eclipse.lsat.common.scheduler.graph.Task;
import org.eclipse.lsat.common.scheduler.graph.TaskDependencyGraph;
import org.eclipse.lsat.common.scheduler.graph.TimeConstraint;
import org.eclipse.lsat.common.scheduler.graph.util.ConstraintsCollector;
import org.eclipse.lsat.common.scheduler.graph.util.ResampleTimeStepData;
import org.eclipse.lsat.common.scheduler.resources.Resource;
import org.eclipse.lsat.common.scheduler.resources.ResourceContainer;
import org.eclipse.lsat.common.scheduler.schedule.DependencyBoundary;
import org.eclipse.lsat.common.scheduler.schedule.Schedule;
import org.eclipse.lsat.common.scheduler.schedule.ScheduledDependency;
import org.eclipse.lsat.common.scheduler.schedule.ScheduledDependencyType;
import org.eclipse.lsat.common.scheduler.schedule.ScheduledTask;
import org.eclipse.lsat.common.scheduler.schedule.Sequence;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function0;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;
import org.eclipse.xtext.xbase.lib.Pair;
import org.eclipse.xtext.xbase.lib.XbaseGenerated;

@SuppressWarnings("all")
public class GenerateAll {
  private static final String PASSING_MOVE_TIMING_GAP = "PassingMoveTimingGap";

  private static final DecimalFormat TIME_FORMATTER = new DecimalFormat("0.######");

  private final TaskDependencyGraph<Task> graph;

  private final Schedule<Task> schedule;

  private final IContainer targetIContainer;

  private final String[] arguments;

  private boolean erroneousPassingMoveColorScheme;

  private boolean criticalPathColorScheme;

  private final boolean stochasticImpactColorScheme;

  private final double motionPlotsSampleFrequency;

  private int etfTimeSeriesCounter;

  /**
   * Constructor.
   * 
   * @param modelURI
   *            is the URI of the model.
   * @param targetFolder
   *            is the output folder
   * @param arguments
   *            are the other arguments
   * @throws IOException
   *             Thrown when the output cannot be saved.
   * @generated NOT
   */
  public GenerateAll(final TaskDependencyGraph<Task> graph, final Schedule<Task> schedule, final IContainer targetIContainer, final boolean criticalPathColorScheme, final boolean stochasticImpactColorScheme, final double motionPlotsSampleFrequency, final String... arguments) {
    this.graph = graph;
    this.schedule = schedule;
    this.targetIContainer = targetIContainer;
    this.arguments = arguments;
    final Function1<Aspect<ScheduledTask<Task>, ScheduledDependency>, Boolean> _function = (Aspect<ScheduledTask<Task>, ScheduledDependency> it) -> {
      String _name = it.getName();
      return Boolean.valueOf(Objects.equals(_name, GenerateAll.PASSING_MOVE_TIMING_GAP));
    };
    boolean _isEmpty = IterableExtensions.isEmpty(IterableExtensions.<Aspect<ScheduledTask<Task>, ScheduledDependency>>filter(schedule.getAspects(), _function));
    boolean _not = (!_isEmpty);
    this.erroneousPassingMoveColorScheme = _not;
    this.criticalPathColorScheme = ((!this.erroneousPassingMoveColorScheme) && criticalPathColorScheme);
    this.stochasticImpactColorScheme = ((!this.erroneousPassingMoveColorScheme) && stochasticImpactColorScheme);
    this.motionPlotsSampleFrequency = motionPlotsSampleFrequency;
  }

  /**
   * Launches the generation.
   * 
   * @param monitor
   *            This will be used to display progress information to the user.
   * @throws IOException
   *             Thrown when the output cannot be saved.
   * @generated NOT
   */
  public IFile generate(final IProgressMonitor monitor) throws CoreException, IOException {
    StringConcatenation _builder = new StringConcatenation();
    String _elvis = null;
    String _head = null;
    if (((Iterable<String>)Conversions.doWrapArray(this.arguments))!=null) {
      _head=IterableExtensions.<String>head(((Iterable<String>)Conversions.doWrapArray(this.arguments)));
    }
    if (_head != null) {
      _elvis = _head;
    } else {
      String _name = this.schedule.getName();
      _elvis = _name;
    }
    _builder.append(_elvis);
    _builder.append(".etf");
    final String etfFileString = _builder.toString();
    Path _path = new Path(etfFileString);
    final IFile etfFile = this.targetIContainer.getFile(_path);
    try (final ByteArrayInputStream contents = new Function0<ByteArrayInputStream>() {
      @Override
      public ByteArrayInputStream apply() {
        try {
          byte[] _bytes = GenerateAll.this.generateETF(GenerateAll.this.schedule).getBytes(etfFile.getCharset(true));
          return new ByteArrayInputStream(_bytes);
        } catch (Throwable _e) {
          throw Exceptions.sneakyThrow(_e);
        }
      }
    }.apply()) {
      boolean _exists = etfFile.exists();
      if (_exists) {
        etfFile.setContents(contents, true, true, monitor);
      } else {
        etfFile.create(contents, true, monitor);
      }
    }
    StringConcatenation _builder_1 = new StringConcatenation();
    _builder_1.append(".");
    _builder_1.append(etfFileString);
    _builder_1.append(".view");
    Path _path_1 = new Path(_builder_1.toString());
    final IFile confFile = this.targetIContainer.getFile(_path_1);
    try (final ByteArrayInputStream contents = new Function0<ByteArrayInputStream>() {
      @Override
      public ByteArrayInputStream apply() {
        try {
          byte[] _bytes = GenerateAll.this.generateTraceViewConfiguration().getBytes(confFile.getCharset(true));
          return new ByteArrayInputStream(_bytes);
        } catch (Throwable _e) {
          throw Exceptions.sneakyThrow(_e);
        }
      }
    }.apply()) {
      boolean _exists = confFile.exists();
      boolean _not = (!_exists);
      if (_not) {
        confFile.create(contents, true, monitor);
      }
    }
    return etfFile;
  }

  public String generateETF(final Schedule<? extends Task> schedule) {
    final Function1<Sequence<? extends Task>, Boolean> _function = (Sequence<? extends Task> it) -> {
      boolean _isEvent = this.isEvent(it);
      return Boolean.valueOf((!_isEvent));
    };
    final List<? extends Sequence<? extends Task>> sequences = IterableExtensions.toList(IterableExtensions.filter(schedule.getSequences(), _function));
    final Function1<Sequence<? extends Task>, Boolean> _function_1 = (Sequence<? extends Task> it) -> {
      return Boolean.valueOf(this.isEvent(it));
    };
    final Iterable<? extends Sequence<? extends Task>> eventSequences = IterableExtensions.filter(schedule.getSequences(), _function_1);
    final Function1<Pair<Integer, Sequence<? extends Task>>, Sequence<? extends Task>> _function_2 = (Pair<Integer, Sequence<? extends Task>> it) -> {
      return it.getValue();
    };
    final Function1<Pair<Integer, Sequence<? extends Task>>, Integer> _function_3 = (Pair<Integer, Sequence<? extends Task>> it) -> {
      return it.getKey();
    };
    final Map<Sequence<? extends Task>, Integer> sequenceIds = IterableExtensions.<Pair<Integer, Sequence<? extends Task>>, Sequence<? extends Task>, Integer>toMap(IterableExtensions.<Sequence<? extends Task>>indexed(sequences), _function_2, _function_3);
    final Function1<Sequence<? extends Task>, List<ScheduledTask<?>>> _function_4 = (Sequence<? extends Task> it) -> {
      EList<? extends ScheduledTask<? extends Task>> _scheduledTasks = it.getScheduledTasks();
      return ((List<ScheduledTask<?>>) _scheduledTasks);
    };
    final Iterable<ScheduledTask<?>> tasks = IterableExtensions.flatMap(schedule.getSequences(), _function_4);
    final Function1<Pair<Integer, ScheduledTask<?>>, ScheduledTask<?>> _function_5 = (Pair<Integer, ScheduledTask<?>> it) -> {
      return it.getValue();
    };
    final Function1<Pair<Integer, ScheduledTask<?>>, Integer> _function_6 = (Pair<Integer, ScheduledTask<?>> it) -> {
      return it.getKey();
    };
    final Map<ScheduledTask<?>, Integer> taskIds = IterableExtensions.<Pair<Integer, ScheduledTask<?>>, ScheduledTask<?>, Integer>toMap(IterableExtensions.<ScheduledTask<?>>indexed(tasks), _function_5, _function_6);
    final Function1<ScheduledTask<?>, Boolean> _function_7 = (ScheduledTask<?> it) -> {
      return Boolean.valueOf(this.isEvent(it));
    };
    final Function1<ScheduledTask<?>, String> _function_8 = (ScheduledTask<?> it) -> {
      return this.getEventName(it);
    };
    final List<String> uniqueEvents = IterableExtensions.<String>toList(IterableExtensions.<String>sort(IterableExtensions.<String>toSet(IterableExtensions.<ScheduledTask<?>, String>map(IterableExtensions.<ScheduledTask<?>>filter(tasks, _function_7), _function_8))));
    final Function1<ScheduledTask<?>, List<Edge>> _function_9 = (ScheduledTask<?> it) -> {
      final Function1<Edge, Integer> _function_10 = (Edge it_1) -> {
        return taskIds.get(it_1.getTargetNode());
      };
      return IterableExtensions.<Edge, Integer>sortBy(it.getOutgoingEdges(), _function_10);
    };
    final Function1<ScheduledDependency, Boolean> _function_10 = (ScheduledDependency it) -> {
      Dependency _dependency = it.getDependency();
      return Boolean.valueOf((_dependency == null));
    };
    final Iterable<ScheduledDependency> dependencies = IterableExtensions.<ScheduledDependency>reject(Iterables.<ScheduledDependency>filter(IterableExtensions.<ScheduledTask<?>, Edge>flatMap(tasks, _function_9), ScheduledDependency.class), _function_10);
    final Function1<Pair<Integer, ScheduledDependency>, ScheduledDependency> _function_11 = (Pair<Integer, ScheduledDependency> it) -> {
      return it.getValue();
    };
    final Function1<Pair<Integer, ScheduledDependency>, Integer> _function_12 = (Pair<Integer, ScheduledDependency> it) -> {
      return it.getKey();
    };
    final Map<ScheduledDependency, Integer> dependencyIds = IterableExtensions.<Pair<Integer, ScheduledDependency>, ScheduledDependency, Integer>toMap(IterableExtensions.<ScheduledDependency>indexed(dependencies), _function_11, _function_12);
    final List<Constraint> constraints = ConstraintsCollector.<Task>collectAllConstraints(this.graph);
    final Function1<Pair<Integer, ScheduledTask<?>>, Task> _function_13 = (Pair<Integer, ScheduledTask<?>> it) -> {
      return it.getValue().getTask();
    };
    final Function1<Pair<Integer, ScheduledTask<?>>, Integer> _function_14 = (Pair<Integer, ScheduledTask<?>> it) -> {
      return it.getKey();
    };
    final Map<Task, Integer> constraintTaskIds = IterableExtensions.<Pair<Integer, ScheduledTask<?>>, Task, Integer>toMap(IterableExtensions.<ScheduledTask<?>>indexed(tasks), _function_13, _function_14);
    final Function1<Pair<Integer, Constraint>, Constraint> _function_15 = (Pair<Integer, Constraint> it) -> {
      return it.getValue();
    };
    final Function1<Pair<Integer, Constraint>, Integer> _function_16 = (Pair<Integer, Constraint> it) -> {
      Integer _key = it.getKey();
      int _size = dependencyIds.size();
      return Integer.valueOf(((_key).intValue() + _size));
    };
    final Map<Constraint, Integer> constraintIds = IterableExtensions.<Pair<Integer, Constraint>, Constraint, Integer>toMap(IterableExtensions.<Constraint>indexed(constraints), _function_15, _function_16);
    this.etfTimeSeriesCounter = IterableExtensions.size(tasks);
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("TU SECONDS");
    _builder.newLine();
    _builder.append("O 0");
    _builder.newLine();
    _builder.append("T");
    _builder.newLine();
    {
      for(final Sequence<? extends Task> sequence : sequences) {
        _builder.append("R ");
        Integer _get = sequenceIds.get(sequence);
        _builder.append(_get);
        _builder.append(" 100.0 true;");
        String _attributes = this.getAttributes(sequence);
        _builder.append(_attributes);
        _builder.newLineIfNotEmpty();
      }
    }
    {
      boolean _isEmpty = IterableExtensions.isEmpty(eventSequences);
      boolean _not = (!_isEmpty);
      if (_not) {
        _builder.append("R ");
        int _size = sequences.size();
        _builder.append(_size);
        _builder.append(" 100.0 true;name=->Events,type=Event");
        _builder.newLineIfNotEmpty();
      }
    }
    {
      for(final ScheduledTask<?> task : tasks) {
        _builder.append("C ");
        Integer _get_1 = taskIds.get(task);
        _builder.append(_get_1);
        _builder.append(" ");
        String _format = GenerateAll.TIME_FORMATTER.format(task.getStartTime());
        _builder.append(_format);
        _builder.append(" ");
        String _format_1 = GenerateAll.TIME_FORMATTER.format(task.getEndTime());
        _builder.append(_format_1);
        _builder.append(" ");
        Integer _sequenceId = this.getSequenceId(task, sequenceIds);
        _builder.append(_sequenceId);
        _builder.append(" ");
        String _offsetAndAmount = this.getOffsetAndAmount(task, uniqueEvents);
        _builder.append(_offsetAndAmount);
        _builder.append(";");
        String _attributes_1 = this.getAttributes(task);
        _builder.append(_attributes_1);
        _builder.newLineIfNotEmpty();
      }
    }
    {
      for(final ScheduledDependency dependency : dependencies) {
        _builder.append("D ");
        Integer _get_2 = dependencyIds.get(dependency);
        _builder.append(_get_2);
        _builder.append(" ");
        int _etfType = this.getEtfType(dependency);
        _builder.append(_etfType);
        _builder.append(" ");
        Integer _get_3 = taskIds.get(dependency.getSourceNode());
        _builder.append(_get_3);
        _builder.append(" ");
        Integer _get_4 = taskIds.get(dependency.getTargetNode());
        _builder.append(_get_4);
        _builder.append(";");
        String _attributes_2 = this.getAttributes(dependency);
        _builder.append(_attributes_2);
        _builder.newLineIfNotEmpty();
      }
    }
    {
      for(final Constraint constraint : constraints) {
        _builder.append("D ");
        Integer _get_5 = constraintIds.get(constraint);
        _builder.append(_get_5);
        _builder.append(" 2 ");
        Integer _get_6 = constraintTaskIds.get(constraint.getSource());
        _builder.append(_get_6);
        _builder.append(" ");
        Integer _get_7 = constraintTaskIds.get(constraint.getTarget());
        _builder.append(_get_7);
        _builder.append(";");
        String _attributes_3 = this.getAttributes(constraint);
        _builder.append(_attributes_3);
        _builder.newLineIfNotEmpty();
      }
    }
    {
      for(final Sequence<? extends Task> sequence_1 : sequences) {
        String _join = IterableExtensions.join(this.writeTimeSeries(this.getTimeSamples(sequence_1)), "\n");
        _builder.append(_join);
        _builder.newLineIfNotEmpty();
      }
    }
    return _builder.toString();
  }

  public String generateTraceViewConfiguration() {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("activityView : false");
    _builder.newLine();
    _builder.append("sortAlphabetic : false");
    _builder.newLine();
    _builder.append("showClaimLabels : true");
    _builder.newLine();
    _builder.append("claimColoring : color");
    _builder.newLine();
    _builder.append("claimDescribing: name");
    _builder.newLine();
    _builder.append("claimGrouping : description");
    _builder.newLine();
    _builder.append("dependencyColoring : color");
    _builder.newLine();
    _builder.append("dependencyFiltering :  { boundary = Between peripherals within 1 resource , Between peripherals in different resources ; constraint = JIT constraint , Time constraint} ");
    _builder.newLine();
    _builder.append("resourceDescribing: name");
    _builder.newLine();
    _builder.append("resourceFiltering :  { type = Peripheral , Resource, Event }");
    _builder.newLine();
    _builder.append("timeSeriesColoring : axis");
    _builder.newLine();
    _builder.append("showTimeSeries : false");
    _builder.newLine();
    return _builder.toString();
  }

  private LinkedHashMap<String, LinkedHashMap<String, List<Pair<BigDecimal, BigDecimal>>>> getTimeSamples(final Sequence<? extends Task> sequence) {
    final LinkedHashMap<String, LinkedHashMap<String, List<Pair<BigDecimal, BigDecimal>>>> result = new LinkedHashMap<String, LinkedHashMap<String, List<Pair<BigDecimal, BigDecimal>>>>();
    EList<? extends ScheduledTask<? extends Task>> _scheduledTasks = sequence.getScheduledTasks();
    for (final ScheduledTask<? extends Task> task : _scheduledTasks) {
      {
        final Map<String, List<List<BigDecimal>>> resampledExecutionData = ResampleTimeStepData.resample(task.getTask().getExecutionData(), this.motionPlotsSampleFrequency);
        ArrayList<String> _arrayList = new ArrayList<String>();
        List<String> parameters = ((List<String>) _arrayList);
        int _size = task.getTask().getExecutionDataParameters().size();
        boolean _greaterThan = (_size > 0);
        if (_greaterThan) {
          parameters = task.getTask().getExecutionDataParameters().subList(1, task.getTask().getExecutionDataParameters().size());
        }
        Iterable<Pair<Integer, String>> _indexed = IterableExtensions.<String>indexed(parameters);
        for (final Pair<Integer, String> indexedParam : _indexed) {
          {
            Integer _key = indexedParam.getKey();
            final int parameterIndex = ((_key).intValue() + 1);
            final String parameterName = indexedParam.getValue();
            StringConcatenation _builder = new StringConcatenation();
            String _etfName = this.getEtfName(sequence.getResource());
            _builder.append(_etfName);
            _builder.append(" ");
            _builder.append(parameterName);
            final String resultName = _builder.toString();
            final Function<String, LinkedHashMap<String, List<Pair<BigDecimal, BigDecimal>>>> _function = (String it) -> {
              return new LinkedHashMap<String, List<Pair<BigDecimal, BigDecimal>>>();
            };
            final LinkedHashMap<String, List<Pair<BigDecimal, BigDecimal>>> resultParam = result.computeIfAbsent(resultName, _function);
            List<String> _sort = IterableExtensions.<String>sort(resampledExecutionData.keySet());
            for (final String setpoint : _sort) {
              {
                final List<List<BigDecimal>> timeStepData = resampledExecutionData.get(setpoint);
                if (((timeStepData.size() > 0) && (parameterIndex < timeStepData.get(0).size()))) {
                  final Function<String, List<Pair<BigDecimal, BigDecimal>>> _function_1 = (String it) -> {
                    return new ArrayList<Pair<BigDecimal, BigDecimal>>();
                  };
                  final List<Pair<BigDecimal, BigDecimal>> resultParamSetPoint = resultParam.computeIfAbsent(setpoint, _function_1);
                  Pair<BigDecimal, BigDecimal> lastPair = ((Pair<BigDecimal, BigDecimal>) null);
                  BigDecimal lastData = ((BigDecimal) null);
                  for (final List<BigDecimal> timeStep : timeStepData) {
                    {
                      BigDecimal _get = timeStep.get(0);
                      BigDecimal _startTime = task.getStartTime();
                      final BigDecimal t = _get.add(_startTime).setScale(6, RoundingMode.HALF_UP);
                      final BigDecimal d = timeStep.get(parameterIndex).setScale(6, RoundingMode.HALF_UP);
                      boolean _notEquals = (!Objects.equals(d, lastData));
                      if (_notEquals) {
                        if ((lastPair != null)) {
                          if ((resultParamSetPoint.isEmpty() || (!Objects.equals(resultParamSetPoint.getLast().getKey(), lastPair.getKey())))) {
                            resultParamSetPoint.add(lastPair);
                          }
                          lastPair = null;
                        }
                        if ((resultParamSetPoint.isEmpty() || (!Objects.equals(resultParamSetPoint.getLast().getKey(), t)))) {
                          Pair<BigDecimal, BigDecimal> _pair = new Pair<BigDecimal, BigDecimal>(t, d);
                          resultParamSetPoint.add(_pair);
                        }
                        lastData = d;
                      } else {
                        Pair<BigDecimal, BigDecimal> _pair_1 = new Pair<BigDecimal, BigDecimal>(t, d);
                        lastPair = _pair_1;
                      }
                    }
                  }
                  if ((lastPair != null)) {
                    if ((resultParamSetPoint.isEmpty() || (!Objects.equals(resultParamSetPoint.getLast().getKey(), lastPair.getKey())))) {
                      resultParamSetPoint.add(lastPair);
                    }
                  }
                }
              }
            }
          }
        }
      }
    }
    return result;
  }

  private ArrayList<String> writeTimeSeries(final LinkedHashMap<String, LinkedHashMap<String, List<Pair<BigDecimal, BigDecimal>>>> timeSamples) {
    final ArrayList<String> result = new ArrayList<String>();
    final Function1<Map.Entry<String, LinkedHashMap<String, List<Pair<BigDecimal, BigDecimal>>>>, Boolean> _function = (Map.Entry<String, LinkedHashMap<String, List<Pair<BigDecimal, BigDecimal>>>> it) -> {
      boolean _isEmpty = it.getValue().isEmpty();
      return Boolean.valueOf((!_isEmpty));
    };
    Iterable<Map.Entry<String, LinkedHashMap<String, List<Pair<BigDecimal, BigDecimal>>>>> _filter = IterableExtensions.<Map.Entry<String, LinkedHashMap<String, List<Pair<BigDecimal, BigDecimal>>>>>filter(timeSamples.entrySet(), _function);
    for (final Map.Entry<String, LinkedHashMap<String, List<Pair<BigDecimal, BigDecimal>>>> parameter : _filter) {
      {
        int seriesCounter = 0;
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("TS ");
        _builder.append(this.etfTimeSeriesCounter);
        _builder.append(" ; name=");
        String _key = parameter.getKey();
        _builder.append(_key);
        result.add(_builder.toString());
        Set<Map.Entry<String, List<Pair<BigDecimal, BigDecimal>>>> _entrySet = parameter.getValue().entrySet();
        for (final Map.Entry<String, List<Pair<BigDecimal, BigDecimal>>> axis : _entrySet) {
          {
            StringConcatenation _builder_1 = new StringConcatenation();
            _builder_1.append("TSS ");
            _builder_1.append(this.etfTimeSeriesCounter);
            _builder_1.append(" ");
            _builder_1.append(seriesCounter);
            _builder_1.append("; name=");
            String _key_1 = parameter.getKey();
            _builder_1.append(_key_1);
            _builder_1.append(" ");
            String _key_2 = axis.getKey();
            _builder_1.append(_key_2);
            _builder_1.append(", axis=");
            String _key_3 = axis.getKey();
            _builder_1.append(_key_3);
            result.add(_builder_1.toString());
            int index = 0;
            while ((index < axis.getValue().size())) {
              {
                index = this.writeTimeSeriesPoints(result, this.etfTimeSeriesCounter, seriesCounter, axis.getValue(), index);
                index = this.writeTimeSeriesRange(result, this.etfTimeSeriesCounter, seriesCounter, axis.getValue(), index);
              }
            }
            int _seriesCounter = seriesCounter;
            seriesCounter = (_seriesCounter + 1);
          }
        }
        int _etfTimeSeriesCounter = this.etfTimeSeriesCounter;
        this.etfTimeSeriesCounter = (_etfTimeSeriesCounter + 1);
      }
    }
    return result;
  }

  private boolean checkIfTimeSeriesIsRange(final List<Pair<BigDecimal, BigDecimal>> data, final int startIndex) {
    final int N = 5;
    BigDecimal lastT = data.get(startIndex).getKey().setScale(6, RoundingMode.HALF_UP);
    BigDecimal timeStep = ((BigDecimal) null);
    int _size = data.size();
    boolean sameStep = ((startIndex + N) <= _size);
    for (int i = (startIndex + 1); (((i < (startIndex + N)) && (i < data.size())) && sameStep); i++) {
      {
        BigDecimal t = data.get(i).getKey().setScale(6, RoundingMode.HALF_UP);
        if ((timeStep == null)) {
          timeStep = t.subtract(lastT);
        }
        BigDecimal _subtract = t.subtract(lastT);
        boolean _equals = Objects.equals(_subtract, timeStep);
        sameStep = _equals;
        lastT = t;
      }
    }
    return sameStep;
  }

  private int writeTimeSeriesPoints(final List<String> result, final int timeSeriesId, final int seriesId, final List<Pair<BigDecimal, BigDecimal>> data, final int startIndex) {
    String str = "";
    int index = startIndex;
    boolean writeAsRange = false;
    while (((index < data.size()) && (!writeAsRange))) {
      {
        writeAsRange = this.checkIfTimeSeriesIsRange(data, index);
        if ((!writeAsRange)) {
          final BigDecimal t = data.get(index).getKey().setScale(6, RoundingMode.HALF_UP);
          final BigDecimal v = data.get(index).getValue().setScale(6, RoundingMode.HALF_UP);
          String _str = str;
          StringConcatenation _builder = new StringConcatenation();
          _builder.append(" ");
          _builder.append(t, " ");
          _builder.append(",");
          _builder.append(v, " ");
          str = (_str + _builder);
          index++;
        }
      }
    }
    int _length = str.length();
    boolean _greaterThan = (_length > 0);
    if (_greaterThan) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("TSP ");
      _builder.append(timeSeriesId);
      _builder.append(" ");
      _builder.append(seriesId);
      String _plus = (_builder.toString() + str);
      str = _plus;
      result.add(str);
    }
    return index;
  }

  private int writeTimeSeriesRange(final List<String> result, final int timeSeriesId, final int seriesId, final List<Pair<BigDecimal, BigDecimal>> data, final int startIndex) {
    int _size = data.size();
    boolean _greaterEqualsThan = ((startIndex + 1) >= _size);
    if (_greaterEqualsThan) {
      return startIndex;
    }
    final BigDecimal t0 = data.get(startIndex).getKey().setScale(6, RoundingMode.HALF_UP);
    final BigDecimal t1 = data.get((startIndex + 1)).getKey().setScale(6, RoundingMode.HALF_UP);
    final BigDecimal timeStep = t1.subtract(t0);
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("TSR ");
    _builder.append(timeSeriesId);
    _builder.append(" ");
    _builder.append(seriesId);
    _builder.append(" ");
    _builder.append(t0);
    _builder.append(":");
    _builder.append(timeStep);
    _builder.append(" ");
    BigDecimal _setScale = data.get(startIndex).getValue().setScale(6, RoundingMode.HALF_UP);
    _builder.append(_setScale);
    String str = _builder.toString();
    BigDecimal lastT = t0;
    boolean sameStep = true;
    int index = (startIndex + 1);
    while (((index < data.size()) && sameStep)) {
      {
        final BigDecimal t = data.get(index).getKey().setScale(6, RoundingMode.HALF_UP);
        BigDecimal _subtract = t.subtract(lastT);
        boolean _equals = Objects.equals(_subtract, timeStep);
        if (_equals) {
          String _str = str;
          StringConcatenation _builder_1 = new StringConcatenation();
          _builder_1.append(" ");
          BigDecimal _setScale_1 = data.get(index).getValue().setScale(6, RoundingMode.HALF_UP);
          _builder_1.append(_setScale_1, " ");
          str = (_str + _builder_1);
          lastT = t;
          index++;
        } else {
          sameStep = false;
        }
      }
    }
    result.add(str);
    return index;
  }

  private String getAttributes(final Sequence<?> sequence) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("name=");
    String _etfName = this.getEtfName(sequence.getResource());
    _builder.append(_etfName);
    _builder.append(",type=");
    String _type = this.getType(sequence.getResource());
    _builder.append(_type);
    _builder.newLineIfNotEmpty();
    return _builder.toString();
  }

  private String _getEtfName(final Resource resource) {
    StringConcatenation _builder = new StringConcatenation();
    String _name = resource.getName();
    _builder.append(_name);
    return _builder.toString();
  }

  private String _getEtfName(final PeripheralResource resource) {
    StringConcatenation _builder = new StringConcatenation();
    ResourceContainer _container = resource.getContainer();
    String _name = null;
    if (_container!=null) {
      _name=_container.getName();
    }
    _builder.append(_name);
    _builder.append(".");
    String _name_1 = resource.getName();
    _builder.append(_name_1);
    return _builder.toString();
  }

  private String getType(final Resource resource) {
    String _switchResult = null;
    boolean _matched = false;
    if (resource instanceof DispatchGroupResource) {
      _matched=true;
      _switchResult = "Dispatch offset";
    }
    if (!_matched) {
      if (resource instanceof PeripheralResource) {
        _matched=true;
        _switchResult = "Peripheral";
      }
    }
    if (!_matched) {
      if (resource instanceof ClaimReleaseResource) {
        _matched=true;
        _switchResult = "Resource";
      }
    }
    if (!_matched) {
      Class<? extends Resource> _class = resource.getClass();
      String _plus = ("Unsupported type: " + _class);
      throw new IllegalArgumentException(_plus);
    }
    return _switchResult;
  }

  private Integer getSequenceId(final ScheduledTask<?> task, final Map<Sequence<? extends Task>, Integer> sequenceIds) {
    boolean _isEvent = this.isEvent(task);
    if (_isEvent) {
      return Integer.valueOf(sequenceIds.size());
    }
    return sequenceIds.get(task.getSequence());
  }

  private String getAttributes(final ScheduledTask<?> task) {
    final LinkedHashMap<String, String> map = CollectionLiterals.<String, String>newLinkedHashMap();
    String _switchResult = null;
    final ScheduledTask<?> it = task;
    boolean _matched = false;
    if (it instanceof ClaimedByScheduledTask) {
      if (this.stochasticImpactColorScheme) {
        _matched=true;
        _switchResult = "NA";
      }
    }
    if (!_matched) {
      if (it instanceof ClaimedByScheduledTask) {
        if (((this.criticalPathColorScheme && this.<Task>isCritical(((ClaimedByScheduledTask)it).getClaims())) && this.<Task>isCritical(((ClaimedByScheduledTask)it).getReleases()))) {
          _matched=true;
          _switchResult = "claim and release";
        }
      }
    }
    if (!_matched) {
      if (it instanceof ClaimedByScheduledTask) {
        if ((this.criticalPathColorScheme && this.<Task>isCritical(((ClaimedByScheduledTask)it).getClaims()))) {
          _matched=true;
          _switchResult = "claim only";
        }
      }
    }
    if (!_matched) {
      if (it instanceof ClaimedByScheduledTask) {
        if ((this.criticalPathColorScheme && this.<Task>isCritical(((ClaimedByScheduledTask)it).getReleases()))) {
          _matched=true;
          _switchResult = "release only";
        }
      }
    }
    if (!_matched) {
      if (it instanceof ClaimedByScheduledTask) {
        if (this.criticalPathColorScheme) {
          _matched=true;
          _switchResult = "none";
        }
      }
    }
    if (!_matched) {
      if (this.stochasticImpactColorScheme) {
        _matched=true;
        StringConcatenation _builder = new StringConcatenation();
        BigDecimal _stochasticCriticality = this.getStochasticCriticality(it);
        BigDecimal _multiply = _stochasticCriticality.multiply(BigDecimal.valueOf(100L));
        _builder.append(_multiply);
        _builder.append("%");
        _switchResult = _builder.toString();
      }
    }
    if (!_matched) {
      if (this.criticalPathColorScheme) {
        _matched=true;
        _switchResult = String.valueOf(this.isCritical(it));
      }
    }
    final String critical = _switchResult;
    this.putConditional(map, "name", this.getDisplayName(task));
    map.put("description", this.getDescription(task));
    this.putConditional(map, "type", this.getType(task));
    this.putConditional(map, "color", this.getColor(task));
    this.putConditional(map, "critical", critical);
    this.putConditional(map, "activity", this.getActivityName(task));
    this.putConditional(map, "baseActivity", this.getBaseActivityName(task));
    final Task tasktask = task.getTask();
    if ((tasktask instanceof ActionTask)) {
      final ResourceAction action = ((ActionTask)tasktask).getAction();
      TracePoint _outerEntry = action.getOuterEntry();
      String _value = null;
      if (_outerEntry!=null) {
        _value=_outerEntry.getValue();
      }
      this.putConditional(map, "outerEntry", _value);
      TracePoint _outerExit = action.getOuterExit();
      String _value_1 = null;
      if (_outerExit!=null) {
        _value_1=_outerExit.getValue();
      }
      this.putConditional(map, "outerExit", _value_1);
      TracePoint _entry = action.getEntry();
      String _value_2 = null;
      if (_entry!=null) {
        _value_2=_entry.getValue();
      }
      this.putConditional(map, "entry", _value_2);
      TracePoint _exit = action.getExit();
      String _value_3 = null;
      if (_exit!=null) {
        _value_3=_exit.getValue();
      }
      this.putConditional(map, "exit", _value_3);
    }
    final Consumer<Map.Entry<String, String>> _function = (Map.Entry<String, String> e) -> {
      map.put(e.getKey(), e.getValue().toString());
    };
    task.getProperties().forEach(_function);
    boolean _isErroneousPassingMove = this.isErroneousPassingMove(task);
    if (_isErroneousPassingMove) {
      map.put("error", "Concatenated move is interrupted");
    }
    this.addStatistics(task, map);
    this.addUserAttributes(task, map);
    final Function1<Map.Entry<String, String>, String> _function_1 = (Map.Entry<String, String> it_1) -> {
      String _trim = it_1.getKey().trim();
      String _plus = (_trim + "=");
      String _escape = this.escape(it_1.getValue());
      return (_plus + _escape);
    };
    return IterableExtensions.join(IterableExtensions.<Map.Entry<String, String>, String>map(map.entrySet(), _function_1), ",");
  }

  private Map<String, String> putConditional(final Map<String, String> map, final String key, final String value) {
    if (((value != null) && (!value.isEmpty()))) {
      map.put(key, value);
    }
    return map;
  }

  private String escape(final String string) {
    String _replaceAll = null;
    if (string!=null) {
      _replaceAll=string.replaceAll("([=,])", "\\\\$1");
    }
    return _replaceAll.trim();
  }

  private String getDisplayName(final ScheduledTask<?> task) {
    return task.getName();
  }

  /**
   * Claim, Release and ClaimedBy tasks are visualized as 'low' bars and all other tasks as 'high' bars.
   */
  private String getOffsetAndAmount(final ScheduledTask<?> task, final List<String> eventNames) {
    boolean _isEvent = this.isEvent(task);
    if (_isEvent) {
      final int index = eventNames.indexOf(this.getEventName(task));
      int _size = eventNames.size();
      int _divide = (100 / _size);
      final int size = Math.min(30, _divide);
      StringConcatenation _builder = new StringConcatenation();
      _builder.append((size * index));
      _builder.append(" ");
      _builder.append(size);
      return _builder.toString();
    }
    String _switchResult = null;
    Task _task = task.getTask();
    boolean _matched = false;
    if (_task instanceof ClaimTask) {
      _matched=true;
    }
    if (!_matched) {
      if (_task instanceof ReleaseTask) {
        _matched=true;
      }
    }
    if (_matched) {
      _switchResult = "0.0 20.0";
    }
    if (!_matched) {
      _switchResult = "20.0 80.0";
    }
    return _switchResult;
  }

  private Dispatch getDispatch(final ScheduledTask<?> task) {
    return this.getDispatch(task.getTask());
  }

  private Dispatch getDispatch(final Task task) {
    final DirectedGraph<?, ?> graph = task.getGraph();
    Dispatch _xifexpression = null;
    if ((graph instanceof DispatchGraph)) {
      _xifexpression = ((DispatchGraph)graph).getDispatch();
    }
    return _xifexpression;
  }

  private String _getActivityName(final ScheduledTask<?> task) {
    Dispatch _dispatch = this.getDispatch(task);
    Activity _activity = null;
    if (_dispatch!=null) {
      _activity=_dispatch.getActivity();
    }
    String _name = null;
    if (_activity!=null) {
      _name=_activity.getName();
    }
    return _name;
  }

  /**
   * return the claim activity name or if there are more then 1 peripheral action
   * the disjunct activities of these actions are returned
   */
  private String _getActivityName(final ClaimedByScheduledTask task) {
    final Function1<ScheduledTask<Task>, String> _function = (ScheduledTask<Task> it) -> {
      Dispatch _dispatch = this.getDispatch(it);
      Activity _activity = null;
      if (_dispatch!=null) {
        _activity=_dispatch.getActivity();
      }
      String _name = null;
      if (_activity!=null) {
        _name=_activity.getName();
      }
      return _name;
    };
    return IterableExtensions.join(ListExtensions.<ScheduledTask<Task>, String>map(task.getClaims(), _function), ", ");
  }

  private String getBaseActivityName(final ScheduledTask<?> task) {
    Dispatch _dispatch = this.getDispatch(task);
    Activity _activity = null;
    if (_dispatch!=null) {
      _activity=_dispatch.getActivity();
    }
    String _originalName = null;
    if (_activity!=null) {
      _originalName=_activity.getOriginalName();
    }
    return _originalName;
  }

  private String getDescription(final ScheduledTask<?> task) {
    String _xifexpression = null;
    Dispatch _dispatch = this.getDispatch(task);
    boolean _tripleNotEquals = (_dispatch != null);
    if (_tripleNotEquals) {
      String _elvis = null;
      String _elvis_1 = null;
      String _description = this.getDispatch(task).getDescription();
      if (_description != null) {
        _elvis_1 = _description;
      } else {
        String _descriptionFromUserAttributes = this.getDescriptionFromUserAttributes(this.getDispatch(task));
        _elvis_1 = _descriptionFromUserAttributes;
      }
      if (_elvis_1 != null) {
        _elvis = _elvis_1;
      } else {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("Activity: ");
        String _activityName = this.getActivityName(task);
        _builder.append(_activityName);
        _elvis = _builder.toString();
      }
      _xifexpression = _elvis;
    } else {
      _xifexpression = "";
    }
    return _xifexpression;
  }

  private String getDescriptionFromUserAttributes(final EObject object) {
    if ((object instanceof HasUserAttributes)) {
      final Function1<Map.Entry<Attribute, String>, Boolean> _function = (Map.Entry<Attribute, String> it) -> {
        String _name = it.getKey().getName();
        return Boolean.valueOf(Objects.equals(_name, "description"));
      };
      final Map.Entry<Attribute, String> entry = IterableExtensions.<Map.Entry<Attribute, String>>findFirst(((HasUserAttributes)object).getUserAttributes(), _function);
      if ((entry != null)) {
        return entry.getValue();
      }
    }
    if ((object != null)) {
      return this.getDescriptionFromUserAttributes(object.eContainer());
    }
    return null;
  }

  private void addUserAttributes(final EObject object, final Map<String, String> map) {
    if ((object != null)) {
      this.addUserAttributes(object.eContainer(), map);
    }
    if ((object instanceof HasUserAttributes)) {
      final Function1<Map.Entry<Attribute, String>, Boolean> _function = (Map.Entry<Attribute, String> it) -> {
        String _name = it.getKey().getName();
        return Boolean.valueOf((!Objects.equals("description", _name)));
      };
      final Consumer<Map.Entry<Attribute, String>> _function_1 = (Map.Entry<Attribute, String> it) -> {
        this.putConditional(map, it.getKey().getName(), it.getValue());
      };
      IterableExtensions.<Map.Entry<Attribute, String>>filter(((HasUserAttributes)object).getUserAttributes(), _function).forEach(_function_1);
    }
  }

  private void addUserAttributes(final ScheduledTask<?> task, final Map<String, String> map) {
    Dispatch _dispatch = this.getDispatch(task);
    EMap<Attribute, String> _userAttributes = null;
    if (_dispatch!=null) {
      _userAttributes=_dispatch.getUserAttributes();
    }
    boolean _tripleNotEquals = (_userAttributes != null);
    if (_tripleNotEquals) {
      this.addUserAttributes(this.getDispatch(task), map);
    }
  }

  private String getType(final ScheduledTask<?> task) {
    String _xifexpression = null;
    boolean _isEvent = this.isEvent(task);
    if (_isEvent) {
      _xifexpression = "Event";
    } else {
      String _xifexpression_1 = null;
      boolean _isClaimOrRelease = this.isClaimOrRelease(task);
      if (_isClaimOrRelease) {
        _xifexpression_1 = "Claim or release";
      } else {
        String _xifexpression_2 = null;
        Task _task = task.getTask();
        if ((_task instanceof DispatchGroupTask)) {
          _xifexpression_2 = "Dispatch offset";
        } else {
          String _xifexpression_3 = null;
          Task _task_1 = task.getTask();
          if ((_task_1 instanceof PeripheralActionTask)) {
            _xifexpression_3 = "Peripheral task";
          } else {
            throw new IllegalArgumentException(("Unsupported task type: " + task));
          }
          _xifexpression_2 = _xifexpression_3;
        }
        _xifexpression_1 = _xifexpression_2;
      }
      _xifexpression = _xifexpression_1;
    }
    return _xifexpression;
  }

  private String getColor(final ScheduledTask<?> task) {
    String _xifexpression = null;
    if (this.erroneousPassingMoveColorScheme) {
      String _xifexpression_1 = null;
      boolean _isErroneousPassingMove = this.isErroneousPassingMove(task);
      if (_isErroneousPassingMove) {
        _xifexpression_1 = "orange";
      } else {
        _xifexpression_1 = "light_gray";
      }
      _xifexpression = _xifexpression_1;
    } else {
      String _xifexpression_2 = null;
      if (this.stochasticImpactColorScheme) {
        String _switchResult = null;
        boolean _matched = false;
        if (task instanceof ClaimedByScheduledTask) {
          _matched=true;
          _switchResult = GanttColor.ClaimReleaseNotCritical.getColor();
        }
        if (!_matched) {
          _switchResult = GanttColor.getStochasticColor(this.getStochasticCriticality(task)).getColor();
        }
        _xifexpression_2 = _switchResult;
      } else {
        String _xifexpression_3 = null;
        if (this.criticalPathColorScheme) {
          String _xifexpression_4 = null;
          if ((task instanceof ClaimedByScheduledTask)) {
            String _switchResult_1 = null;
            boolean _matched_1 = false;
            if ((this.<Task>isCritical(((ClaimedByScheduledTask)task).getClaims()) && this.<Task>isCritical(((ClaimedByScheduledTask)task).getReleases()))) {
              _matched_1=true;
              _switchResult_1 = GanttColor.ClaimAndReleaseCritical.getColor();
            }
            if (!_matched_1) {
              if ((this.<Task>isCritical(((ClaimedByScheduledTask)task).getClaims()) || this.<Task>isCritical(((ClaimedByScheduledTask)task).getReleases()))) {
                _matched_1=true;
                _switchResult_1 = GanttColor.ClaimOrReleaseCritical.getColor();
              }
            }
            if (!_matched_1) {
              _switchResult_1 = GanttColor.ClaimReleaseNotCritical.getColor();
            }
            _xifexpression_4 = _switchResult_1;
          } else {
            String _xifexpression_5 = null;
            boolean _isCritical = this.isCritical(task);
            if (_isCritical) {
              _xifexpression_5 = GanttColor.TaskCritical.getColor();
            } else {
              _xifexpression_5 = GanttColor.TaskNotCritical.getColor();
            }
            _xifexpression_4 = _xifexpression_5;
          }
          _xifexpression_3 = _xifexpression_4;
        } else {
          _xifexpression_3 = this.getColorKey(task);
        }
        _xifexpression_2 = _xifexpression_3;
      }
      _xifexpression = _xifexpression_2;
    }
    return _xifexpression;
  }

  private String getColorKey(final ScheduledTask<?> task) {
    String _xifexpression = null;
    boolean _isEvent = this.isEvent(task);
    if (_isEvent) {
      _xifexpression = this.getEventName(task);
    } else {
      String _xifexpression_1 = null;
      boolean _isClaimOrRelease = this.isClaimOrRelease(task);
      if (_isClaimOrRelease) {
        _xifexpression_1 = this.getDescription(task);
      } else {
        String _xifexpression_2 = null;
        Task _task = task.getTask();
        if ((_task instanceof DispatchGroupTask)) {
          _xifexpression_2 = "Dispatch offset";
        } else {
          String _xifexpression_3 = null;
          Task _task_1 = task.getTask();
          if ((_task_1 instanceof PeripheralActionTask)) {
            Task _task_2 = task.getTask();
            _xifexpression_3 = ((PeripheralActionTask) _task_2).getAction().getResource().fqn();
          } else {
            throw new IllegalArgumentException(("Unsupported task type: " + task));
          }
          _xifexpression_2 = _xifexpression_3;
        }
        _xifexpression_1 = _xifexpression_2;
      }
      _xifexpression = _xifexpression_1;
    }
    return _xifexpression;
  }

  private String getAttributes(final ScheduledDependency dependency) {
    LinkedHashMap<String, String> attrs = CollectionLiterals.<String, String>newLinkedHashMap();
    attrs.put("boundary", this.getEtfBoundary(dependency));
    attrs.put("color", this.getColor(dependency));
    if (this.criticalPathColorScheme) {
      attrs.put("critical", Boolean.valueOf(this.isCritical(dependency)).toString());
    }
    boolean _isEvent = this.isEvent(dependency);
    if (_isEvent) {
      attrs.put("name", this.getEventName(dependency));
    }
    final Function1<Map.Entry<String, String>, String> _function = (Map.Entry<String, String> it) -> {
      String _trim = it.getKey().trim();
      String _plus = (_trim + "=");
      String _escape = this.escape(it.getValue());
      return (_plus + _escape);
    };
    return IterableExtensions.join(IterableExtensions.<Map.Entry<String, String>, String>map(attrs.entrySet(), _function), ",");
  }

  private int getEtfType(final ScheduledDependency dependency) {
    int _switchResult = (int) 0;
    ScheduledDependencyType _type = dependency.getType();
    if (_type != null) {
      switch (_type) {
        case SOURCE_NODE_STARTS_BEFORE_TARGET_NODE_STARTS:
          _switchResult = 0;
          break;
        case SOURCE_NODE_STARTS_BEFORE_TARGET_NODE_ENDS:
          _switchResult = 1;
          break;
        case SOURCE_NODE_ENDS_BEFORE_TARGET_NODE_STARTS:
          _switchResult = 2;
          break;
        case SOURCE_NODE_ENDS_BEFORE_TARGET_NODE_ENDS:
          _switchResult = 3;
          break;
        default:
          break;
      }
    }
    return _switchResult;
  }

  private String getEtfBoundary(final ScheduledDependency dependency) {
    boolean _isEvent = this.isEvent(dependency);
    if (_isEvent) {
      return "Between events";
    }
    String _switchResult = null;
    DependencyBoundary _boundary = dependency.getBoundary();
    if (_boundary != null) {
      switch (_boundary) {
        case CROSS_RESOURCE:
          _switchResult = "Between peripherals within 1 resource";
          break;
        case CROSS_RESOURCE_CONTAINER:
          _switchResult = "Between peripherals in different resources";
          break;
        case IN_RESOURCE:
          _switchResult = "Within 1 peripheral";
          break;
        default:
          break;
      }
    }
    return _switchResult;
  }

  private String getColor(final ScheduledDependency dependency) {
    String _xifexpression = null;
    if (this.erroneousPassingMoveColorScheme) {
      String _xifexpression_1 = null;
      boolean _isEvent = this.isEvent(dependency);
      if (_isEvent) {
        _xifexpression_1 = GanttColor.DependencyEvent.getColor();
      } else {
        _xifexpression_1 = GanttColor.DependencyNormal.getColor();
      }
      _xifexpression = _xifexpression_1;
    } else {
      String _xifexpression_2 = null;
      if (this.criticalPathColorScheme) {
        String _xifexpression_3 = null;
        boolean _isCritical = this.isCritical(dependency);
        if (_isCritical) {
          _xifexpression_3 = GanttColor.DependencyCritical.getColor();
        } else {
          String _xifexpression_4 = null;
          boolean _isEvent_1 = this.isEvent(dependency);
          if (_isEvent_1) {
            _xifexpression_4 = GanttColor.DependencyEvent.getColor();
          } else {
            _xifexpression_4 = GanttColor.DependencyNotCritical.getColor();
          }
          _xifexpression_3 = _xifexpression_4;
        }
        _xifexpression_2 = _xifexpression_3;
      } else {
        String _xifexpression_5 = null;
        boolean _isEvent_2 = this.isEvent(dependency);
        if (_isEvent_2) {
          _xifexpression_5 = GanttColor.DependencyEvent.getColor();
        } else {
          String _xifexpression_6 = null;
          if ((this.isClaimOrRelease(((ScheduledTask<Task>) dependency.getSourceNode())) || this.isClaimOrRelease(((ScheduledTask<Task>) dependency.getTargetNode())))) {
            _xifexpression_6 = GanttColor.DependencyClaimRelease.getColor();
          } else {
            _xifexpression_6 = GanttColor.DependencyNormal.getColor();
          }
          _xifexpression_5 = _xifexpression_6;
        }
        _xifexpression_2 = _xifexpression_5;
      }
      _xifexpression = _xifexpression_2;
    }
    return _xifexpression;
  }

  private String getAttributes(final Constraint constraint) {
    LinkedHashMap<String, String> attrs = CollectionLiterals.<String, String>newLinkedHashMap();
    if ((constraint instanceof JitConstraint)) {
      attrs.put("constraint", "JIT constraint");
      attrs.put("color", "red");
    }
    if ((constraint instanceof TimeConstraint)) {
      attrs.put("constraint", "Time constraint");
      attrs.put("color", "orange");
      attrs.put("lower bound", ((TimeConstraint)constraint).getLowerBound().toString());
      attrs.put("upper bound", ((TimeConstraint)constraint).getUpperBound().toString());
    }
    attrs.put("source", constraint.getSource().getName());
    attrs.put("target", constraint.getTarget().getName());
    final Function1<Map.Entry<String, String>, String> _function = (Map.Entry<String, String> it) -> {
      String _trim = it.getKey().trim();
      String _plus = (_trim + "=");
      String _escape = this.escape(it.getValue());
      return (_plus + _escape);
    };
    return IterableExtensions.join(IterableExtensions.<Map.Entry<String, String>, String>map(attrs.entrySet(), _function), ",");
  }

  private BigDecimal getStochasticCriticality(final ScheduledTask<?> task) {
    BigDecimal _elvis = null;
    StochasticAnnotation _lastOrNull = IterableExtensions.<StochasticAnnotation>lastOrNull(Iterables.<StochasticAnnotation>filter(task.getAspects(), StochasticAnnotation.class));
    BigDecimal _criticality = null;
    if (_lastOrNull!=null) {
      _criticality=_lastOrNull.getCriticality();
    }
    if (_criticality != null) {
      _elvis = _criticality;
    } else {
      _elvis = BigDecimal.ZERO;
    }
    return _elvis;
  }

  private void addStatistics(final ScheduledTask<?> task, final Map<String, String> map) {
    final StochasticAnnotation ann = IterableExtensions.<StochasticAnnotation>lastOrNull(Iterables.<StochasticAnnotation>filter(task.getAspects(), StochasticAnnotation.class));
    if ((ann != null)) {
      final Bounds bounds = ann.getConfidenceInterval();
      if ((bounds != null)) {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("[");
        String _percentage = GenerateAll.toPercentage(bounds.getLower());
        _builder.append(_percentage);
        _builder.append(",");
        String _percentage_1 = GenerateAll.toPercentage(bounds.getUpper());
        _builder.append(_percentage_1);
        _builder.append("]");
        map.put("95% confidence interval", _builder.toString());
      }
      map.put("start time(min)", GenerateAll.TIME_FORMATTER.format(ann.getMin()));
      map.put("start time(mean)", GenerateAll.TIME_FORMATTER.format(ann.getMean()));
      map.put("start time(max)", GenerateAll.TIME_FORMATTER.format(ann.getMax()));
      BigDecimal _skewness = ann.getSkewness();
      boolean _tripleNotEquals = (_skewness != null);
      if (_tripleNotEquals) {
        map.put("start time(skewness)", GenerateAll.TIME_FORMATTER.format(ann.getSkewness()));
      }
      map.put("start time(standard deviation)", GenerateAll.TIME_FORMATTER.format(ann.getStandardDeviation()));
    }
  }

  private <T extends Task> boolean isCritical(final List<ScheduledTask<T>> tasks) {
    final Function1<ScheduledTask<T>, Boolean> _function = (ScheduledTask<T> it) -> {
      return Boolean.valueOf(this.isCritical(it));
    };
    return IterableExtensions.<ScheduledTask<T>>exists(tasks, _function);
  }

  private boolean isCritical(final ScheduledTask<?> task) {
    return ((task != null) && (!IterableExtensions.isEmpty(IterableExtensions.<Aspect<?, ?>>filter(task.getAspects(), ((Function1<Aspect<?, ?>, Boolean>) (Aspect<?, ?> it) -> {
      String _name = it.getName();
      return Boolean.valueOf(Objects.equals(_name, "Critical"));
    })))));
  }

  private boolean isErroneousPassingMove(final ScheduledTask<?> task) {
    return ((task != null) && (!IterableExtensions.isEmpty(IterableExtensions.<Aspect<?, ?>>filter(task.getAspects(), ((Function1<Aspect<?, ?>, Boolean>) (Aspect<?, ?> it) -> {
      String _name = it.getName();
      return Boolean.valueOf(Objects.equals(_name, "PassingMoveTimingGap"));
    })))));
  }

  private boolean isCritical(final ScheduledDependency dependency) {
    final Function1<Aspect<?, ?>, Boolean> _function = (Aspect<?, ?> it) -> {
      String _name = it.getName();
      return Boolean.valueOf(Objects.equals(_name, "Critical"));
    };
    boolean _isEmpty = IterableExtensions.isEmpty(IterableExtensions.<Aspect<?, ?>>filter(dependency.getAspects(), _function));
    return (!_isEmpty);
  }

  private boolean isEvent(final ScheduledDependency dependency) {
    return (this.isEvent(((ScheduledTask<?>) dependency.getSourceNode())) || this.isEvent(((ScheduledTask<?>) dependency.getTargetNode())));
  }

  private boolean isEvent(final ScheduledTask<?> task) {
    boolean _isEmpty = IterableExtensions.isEmpty(Iterables.<EventAnnotation>filter(task.getTask().getAspects(), EventAnnotation.class));
    return (!_isEmpty);
  }

  private <T extends Task> boolean isEvent(final Sequence<T> sequence) {
    final Function1<ScheduledTask<T>, Boolean> _function = (ScheduledTask<T> it) -> {
      return Boolean.valueOf(this.isEvent(it));
    };
    return IterableExtensions.<ScheduledTask<T>>exists(sequence.getScheduledTasks(), _function);
  }

  private String getEventName(final ScheduledTask<?> task) {
    EventAnnotation _head = IterableExtensions.<EventAnnotation>head(Iterables.<EventAnnotation>filter(task.getTask().getAspects(), EventAnnotation.class));
    String _name = null;
    if (_head!=null) {
      _name=_head.getName();
    }
    return _name;
  }

  private String getEventName(final ScheduledDependency dependency) {
    String _xifexpression = null;
    Node _sourceNode = dependency.getSourceNode();
    boolean _isEvent = this.isEvent(((ScheduledTask<?>) _sourceNode));
    if (_isEvent) {
      Node _sourceNode_1 = dependency.getSourceNode();
      _xifexpression = this.getEventName(((ScheduledTask<?>) _sourceNode_1));
    } else {
      String _xifexpression_1 = null;
      Node _targetNode = dependency.getTargetNode();
      boolean _isEvent_1 = this.isEvent(((ScheduledTask<?>) _targetNode));
      if (_isEvent_1) {
        Node _targetNode_1 = dependency.getTargetNode();
        _xifexpression_1 = this.getEventName(((ScheduledTask<?>) _targetNode_1));
      } else {
        _xifexpression_1 = "";
      }
      _xifexpression = _xifexpression_1;
    }
    return _xifexpression;
  }

  private boolean isClaimOrRelease(final ScheduledTask<?> task) {
    return (((task instanceof ClaimedByScheduledTask) || (task.getTask() instanceof ClaimTask)) || (task.getTask() instanceof ReleaseTask));
  }

  private static String toPercentage(final BigDecimal value) {
    if ((value == null)) {
      return "";
    }
    BigDecimal _bigDecimal = new BigDecimal(100);
    final BigDecimal percentage = value.multiply(_bigDecimal).setScale(0, RoundingMode.HALF_UP);
    final String pattern = "0".repeat(2);
    final DecimalFormat formatter = new DecimalFormat(pattern);
    return formatter.format(percentage);
  }

  @XbaseGenerated
  private String getEtfName(final Resource resource) {
    if (resource instanceof PeripheralResource) {
      return _getEtfName((PeripheralResource)resource);
    } else if (resource != null) {
      return _getEtfName(resource);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(resource).toString());
    }
  }

  @XbaseGenerated
  private String getActivityName(final ScheduledTask<?> task) {
    if (task instanceof ClaimedByScheduledTask) {
      return _getActivityName((ClaimedByScheduledTask)task);
    } else if (task != null) {
      return _getActivityName(task);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(task).toString());
    }
  }
}
